<?php

/**
 * tirreno ~ open security analytics
 * Copyright (c) Tirreno Technologies Sàrl (https://www.tirreno.com)
 *
 * Licensed under GNU Affero General Public License version 3 of the or any later version.
 * For full copyright and license information, please see the LICENSE
 * Redistributions of files must retain the above copyright notice.
 *
 * @copyright     Copyright (c) Tirreno Technologies Sàrl (https://www.tirreno.com)
 * @license       https://opensource.org/licenses/AGPL-3.0 AGPL License
 * @link          https://www.tirreno.com Tirreno(tm)
 */

declare(strict_types=1);

namespace Utils;

class Variables {
    private static function getF3(): \Base {
        return \Base::instance();
    }

    public static function getDB(): ?string {
        return getenv('DATABASE_URL') ?: self::getF3()->get('DATABASE_URL');
    }

    public static function getConfigFile(): string {
        return getenv('CONFIG_FILE') ?: 'local/config.local.ini';
    }

    public static function getHosts(): array {
        $env = getenv('SITE');
        $conf = self::getF3()->get('SITE');

        return $env ? explode(',', $env) : (is_array($conf) ? $conf : [$conf]);
    }

    public static function getHost(): string {
        return self::getHosts()[0];
    }

    public static function getAdminEmail(): ?string {
        return getenv('ADMIN_EMAIL') ?: self::getF3()->get('ADMIN_EMAIL');
    }

    public static function getMailLogin(): ?string {
        return getenv('MAIL_LOGIN') ?: self::getF3()->get('MAIL_LOGIN');
    }

    public static function getMailPassword(): ?string {
        return getenv('MAIL_PASS') ?: self::getF3()->get('MAIL_PASS');
    }

    public static function getEnrichmentApi(): string {
        return getenv('ENRICHMENT_API') ?: self::getF3()->get('ENRICHMENT_API');
    }

    public static function getPepper(): string {
        return getenv('PEPPER') ?: self::getF3()->get('PEPPER');
    }

    public static function getLogbookLimit(): int {
        $value = getenv('LOGBOOK_LIMIT') ?: self::getF3()->get('LOGBOOK_LIMIT') ?: \Utils\Constants::get('LOGBOOK_LIMIT');

        return \Utils\Conversion::intValCheckEmpty($value, \Utils\Constants::get('LOGBOOK_LIMIT'));
    }

    public static function getForgotPasswordAllowed(): bool {
        $variable = getenv('ALLOW_FORGOT_PASSWORD') ?: self::getF3()->get('ALLOW_FORGOT_PASSWORD') ?? 'false';

        return filter_var($variable, FILTER_VALIDATE_BOOLEAN, FILTER_NULL_ON_FAILURE) ?? false;
    }

    public static function getEmailPhoneAllowed(): bool {
        $variable = getenv('ALLOW_EMAIL_PHONE') ?: self::getF3()->get('ALLOW_EMAIL_PHONE') ?? 'false';

        return filter_var($variable, FILTER_VALIDATE_BOOLEAN, FILTER_NULL_ON_FAILURE) ?? false;
    }

    public static function getForceHttps(): bool {
        // set 'false' string if FORCE_HTTPS wasn't set due to filter_var() issues
        $variable = getenv('FORCE_HTTPS') ?: self::getF3()->get('FORCE_HTTPS') ?? 'false';

        return filter_var($variable, FILTER_VALIDATE_BOOLEAN, FILTER_NULL_ON_FAILURE) ?? true;
    }

    public static function getHostWithProtocol(): string {
        return (self::getForceHttps() ? 'https://' : 'http://') . self::getHost();
    }

    public static function getHostWithProtocolAndBase(): string {
        return self::getHostWithProtocol() . self::getF3()->get('BASE');
    }

    public static function getAccountOperationQueueBatchSize(): int {
        return \Utils\Conversion::intValCheckEmpty(getenv('ACCOUNT_OPERATION_QUEUE_BATCH_SIZE'), \Utils\Constants::get('ACCOUNT_OPERATION_QUEUE_BATCH_SIZE'));
    }

    public static function getNewEventsBatchSize(): int {
        return \Utils\Conversion::intValCheckEmpty(getenv('NEW_EVENTS_BATCH_SIZE'), \Utils\Constants::get('NEW_EVENTS_BATCH_SIZE'));
    }

    public static function getRuleUsersBatchSize(): int {
        return \Utils\Conversion::intValCheckEmpty(getenv('RULE_USERS_BATCH_SIZE'), \Utils\Constants::get('RULE_USERS_BATCH_SIZE'));
    }

    public static function getAvailableTimezones(): array {
        return array_intersect_key(self::getF3()->get('timezones'), array_flip(\DateTimeZone::listIdentifiers()));
    }

    public static function completedConfig(): bool {
        return
            (getenv('SITE') || self::getF3()->get('SITE')) &&
            (getenv('PEPPER') || self::getF3()->get('PEPPER')) &&
            (getenv('ENRICHMENT_API') || self::getF3()->get('ENRICHMENT_API')) &&
            (getenv('DATABASE_URL') || self::getF3()->get('DATABASE_URL'));
    }
}
